# -*- coding: utf-8 -*-
import logging

import requests

import calculator_hv
import utils
import request
from config import get_config

# important for the sessions to be outside for performance improvements
ernc_session = requests.Session()
water_session = requests.Session()

logging.basicConfig(
    level=logging.ERROR,
    format='[%(asctime)s] %(levelname)s: %(message)s',
)

def H2V_explorer_calculations(PV_CAPEX=None,Wind_CAPEX=None, Ez_CAPEX=None,Ez_eff=None,Ez_stack_life=None):
    coordenadas=(-22.28, -68.46) #estas cooordenadas deben ir dentro de la función y no ser un parámetro. Dejar esto al inicio que sea editable fácilmente

    input_ = get_config(f'inputs/multiple_inputs')
    input_['global']['execution']['position'] = coordenadas
    reference = get_config('constants/input_limits')

    for sub_input_name in input_['inputs']:
        input_[sub_input_name] = get_config(f'inputs/{sub_input_name}')

        logging.info(f'Validating input {sub_input_name} limits')
        year = str(input_[sub_input_name]['project']['year'])
        if year in reference.keys():
            utils.validate_input(reference[year], input_[sub_input_name])
        else:
            logging.warning(f'No reference values for year {year}, skipping validation')

    logging.info('Loading profiles')
    for energy, config in input_['global']['energies'].items():
        if 'profile_template' in config.keys():
            input_['global']['energies'][energy]['profile_config'] = (
                get_config(f'profiles/{config["profile_template"]}'))
            input_['global']['energies'][energy]['profile_config']['position'][0]['lat'] = (
                input_['global']['execution']['position'][0])
            input_['global']['energies'][energy]['profile_config']['position'][0]['lon'] = (
                input_['global']['execution']['position'][1])

    logging.info('Query geoserver')
    if input_['global']['use_local_api']:
        water_api_config = get_config('api/config_api_geo')
    else:
        water_api_config = get_config('api/remote/config_api_geo')
    request_template = get_config('api/geo_request')
    # Computes z coordinate and distance to coast
    request.get_data_water(input_['global'], water_session, water_api_config, request_template)
    # required for model, maybe find another place
    for sub_input_name in input_['inputs']:
        input_[sub_input_name]['project']['position'] = input_['global']['execution']['position']
        input_[sub_input_name]['project']['coord_z'] = input_['global']['execution']['coord_z']
        input_[sub_input_name]['project']['dist_to_coast'] = input_['global']['execution']['dist_to_coast']

    logging.info('Query ernc')
    if input_['global']['use_local_api']:
        ernc_api_config = get_config('api/config_api_ernc')
    else:
        ernc_api_config = get_config('api/remote/config_api_ernc')
    # Computes energy profile per hour
    original_energies_profiles = request.get_data_energies(input_['global'], ernc_session, ernc_api_config)

    # asigns specific run values from function input, overridding json values
    for sub_input_name in input_['inputs']:
        input_[sub_input_name]["energies"]["solar"]["capex"] = PV_CAPEX
        input_[sub_input_name]["energies"]["eolic"]["capex"] = Wind_CAPEX
        input_[sub_input_name]["electrolyzer"]["capex"] = Ez_CAPEX
        input_[sub_input_name]["electrolyzer"]["electrolyzer_efficiency"] = Ez_eff
        input_[sub_input_name]["electrolyzer"]["stack_life"] = Ez_stack_life
    list_list = {}
    lcoes_list = {}
    for sub_input_name in input_['inputs']:
        logging.info(f'Processing {sub_input_name}')
        energies_profiles = original_energies_profiles.copy()
        res_calculation_original = None
        list_lcoh = []
        iterations = 12 if input_['global']['execution']['output_with_energies_balance'] else 1
        for iteration in range(iterations):
            logging.info(f'{iteration:>2d}: Processing inputs')
            factor_energies, lcoes = calculator_hv.process_inputs(input_[sub_input_name], energies_profiles, iteration)

            logging.info(f'{iteration:>2d}: Model calculation')
            res_calculation, lcoh = calculator_hv.run_model(input_[sub_input_name], energies_profiles)

            if iteration == 0:
                res_calculation_original = res_calculation.copy()
            list_lcoh.append((*factor_energies, lcoh))

        list_list[sub_input_name] = list_lcoh
        lcoes_list[sub_input_name] = lcoes
        logging.info(f'Saving output for {sub_input_name}')
        #utils.save_output(input_, sub_input_name, original_energies_profiles, res_calculation_original, list_lcoh)
        logging.info('Done')

    LCOH_solar = list_list[sub_input_name][-1][2]
    LCOH_wind = list_list[sub_input_name][1][2]
    LCOE_solar = lcoes_list[sub_input_name]["solar"]
    LCOE_wind = lcoes_list[sub_input_name]["eolic"]

    lcohs = list_list[sub_input_name][1:]
    lcohs.sort(key=lambda x: x[0])
    lcohs = lcohs[0]

    LCOH_opt = lcohs[2]
    Mix_solar_opt = lcohs[1]
    return {'y1':LCOH_opt,'y2':Mix_solar_opt,'y3':LCOH_solar,'y4':LCOH_wind,'y5':LCOE_solar,'y6':LCOE_wind} # Cada uno de estos valores sería un escalar.
    # Correr todos los cálculos aquí en base a los parámetros PV_CAPEX,Wind_CAPEX, Ez_CAPEX,Ez_eff y Ez_stack_life.



#----------------------- Insumos ----------
# Recordar además proveer
# Coordenadas de cuatro puntos característicos: uno por macrozona (en formato para el modelo). Pueden ser los conversados en la reunión.
# Trayectoria de los valores, que actualmente están siendo considerados en el modelo, para los cinco parámetros de la función.
coordenadas_calama      = (-22.28,-68.46)
coordenadas_sanfernando = (-34.31,-70.54)
coordenadas_balmaceda   = (-45.89,-71.6875)
coordenadas_porvenir    = (-53.21,-70.1)

valores_inputs = {
    "inputs": [
        "2022",
        "2030",
        "2040",
        "2050"
    ],
    "PV_CAPEX": [850, 703, 596, 513],
    "Wind_CAPEX": [1197, 1020, 902, 825],
    "Ez_CAPEX": [1050, 825, 687, 550],
    "Ez_eff": [0.56, 0.66, 0.68, 0.71],
    "Ez_stack_life": [60000, 75000, 100000, 125000],
}

#----------------------Abreviaturas-----------

#Parámetros:
# PV_CAPEX: Costo de inversión en generación solar
# Wind_CAPEX: Costo de inversión en generación eólica
# Ez_CAPEX: Costo de inversión en electrolizadores
# Ez_eff: Eficiencia de electrólisis
# Ez_stack_life: Vida útil del stack del electrólizador

#Métricas relevantes
# LCOH_opt: LCOH en el mix óptimo
# Mix_solar_opt: porcentaje solar en el caso óptimo
# LCOE_opt: LCOE en el mix óptimo # se borra por que no estamos actualmente calculando el LCOE de la energía efectiva proveida en los casos MIX si no que se calcula un LCOE a parte para cada tecnología. 
# LCOH_solar: LCOH para el caso 100% solar
# LCOH_wind: LCOH para el caso 100% eólico
# LCOE_solar: LCOE para el caso 100% solar
# LCOE_wind: LCOE para el caso 100% eólico


if __name__ == "__main__":
    ret = H2V_explorer_calculations(850, 1193, 1050, 0.56, 60000)
    print(ret)
    print("- "*10+"   OK    "+"- "*10)