import numpy as np
import pandas as pd

from config import get_config

constants = get_config('constants/calculator')


def process_inputs(input_, energies_profiles, iteration=0):
    # Modifies input_
    # Adds columns to energies_profiles (Factor Planta)
    factor_eolic = None
    factor_solar = None
    if iteration > 0 and 'eolic' in input_['energies'] and 'solar' in input_['energies']:
        factor_eolic = round((1 - iteration / 10 + 0.1) * 100)
        factor_solar = round((iteration / 10 - 0.1) * 100)

        dimension_eolic = 0.1 * factor_eolic
        dimension_solar = 0.1 * factor_solar
        # always the same btw
        dimension_electrolyzer = (dimension_eolic + dimension_solar) / constants['proportion_energies_electrolyzer']

        input_['energies']['eolic']['dimension'] = dimension_eolic
        input_['energies']['solar']['dimension'] = dimension_solar

        # always 20/3, can be source of numerical divergence as in the input it has only 4 decimals
        input_['electrolyzer']['installed_capacity'] = dimension_electrolyzer

    lcoes = {}
    for energy in input_['energies'].keys():
        if energy in ['eolic', 'solar']:
            energy_config = input_['energies'][energy]
            energies_profiles[f'Factor Planta {energy}'] = (
                    energies_profiles[f'Perfil de Energía - {energy}'] / energy_config['total_power_profile'])

            if energy_config['cost_method'] == 'CAPEX y OPEX' and energy_config['dimension'] > 0:
                energy_config['lcoe'] = (

                        (energy_config['capex'] * energy_config['dimension'] * 1000 +
                         (energy_config['dimension'] * energy_config['capex'] * 1000 * energy_config['opex'] / 0.08) *
                         (1 - 1 / (1 + input_['project']['discount_rate']) ** input_['project']['years_evaluation']))
                        /
                        (energy_config['dimension'] * energies_profiles[f'Factor Planta {energy}'].mean() * 8760 /
                         input_['project']['discount_rate'] *
                         (1 - 1 / (1 + input_['project']['discount_rate']) ** input_['project']['years_evaluation']))

                )
            lcoes[energy] = energy_config["lcoe"]

    return (factor_eolic, factor_solar), lcoes


def calculate_h2_profile(df_h2_profile, electrolyzer_config, energies):
    complem_renewable = {'solar': 'eolic', 'eolic': 'solar'}

    df_h2_profile['Suministro solar'] = 0
    df_h2_profile['Suministro eolic'] = 0
    df_h2_profile['Suministro renovable'] = 0
    df_h2_profile['Suministro grid'] = 0
    df_h2_profile['Total Electricidad'] = 0
    df_h2_profile['Aux Energia solar'] = 0
    df_h2_profile['Aux Energia eolic'] = 0
    df_h2_profile[f'Pérdida Energía solar'] = 0
    df_h2_profile[f'Pérdida Energía eolic'] = 0
    df_h2_profile[f'Suministro solar Real'] = 0
    df_h2_profile[f'Suministro eolic Real'] = 0
    df_h2_profile[f'Suministro grid Real'] = 0
    df_h2_profile['Energía real utilizada'] = 0

    for energy in energies.keys():
        if energy in ['solar', 'eolic']:
            df_h2_profile[f'Suministro {energy}'] = (
                    energies[energy]['dimension'] * df_h2_profile[f'Factor Planta {energy}'])
            df_h2_profile['Suministro renovable'] += df_h2_profile[f'Suministro {energy}']

    if 'grid' in energies.keys():
        grid_complement = electrolyzer_config['installed_capacity'] * energies['grid']['complement_percentage']
        cond_grid_complement = df_h2_profile['Suministro renovable'] < grid_complement
        df_h2_profile.loc[cond_grid_complement, 'Suministro grid'] = (
                grid_complement - df_h2_profile['Suministro renovable'])

    df_h2_profile['Total Electricidad'] = df_h2_profile['Suministro renovable'] + df_h2_profile['Suministro grid']

    cond_total_ge_electro = df_h2_profile['Suministro renovable'] > electrolyzer_config['installed_capacity']
    for energy in energies.keys():
        if energy in ['solar', 'eolic']:
            cond_e_le_electro = df_h2_profile[f'Suministro {energy}'] < electrolyzer_config['installed_capacity']
            df_h2_profile[f'Aux Energia {energy}'] = df_h2_profile[f'Suministro {energy}']
            df_h2_profile.loc[
                cond_total_ge_electro & cond_e_le_electro,
                f'Aux Energia {energy}'
            ] = df_h2_profile[f'Suministro {energy}']
            df_h2_profile.loc[
                cond_total_ge_electro & ~cond_e_le_electro,
                f'Aux Energia {energy}'
            ] = electrolyzer_config['installed_capacity']
            
    for energy in energies.keys():
        if energy in ['solar', 'eolic']:
            cond_lcoe_leq = False
            if complem_renewable[energy] in energies.keys():
                cond_lcoe_leq = energies[energy]['lcoe'] < energies[complem_renewable[energy]]['lcoe']
            residue_electro_aux = (
                electrolyzer_config['installed_capacity'] - df_h2_profile[f'Aux Energia {complem_renewable[energy]}'])
            cond_e_le_residue = df_h2_profile[f'Suministro {energy}'] < residue_electro_aux
            df_h2_profile[f'Suministro {energy} Real'] = df_h2_profile[f'Aux Energia {energy}']
            df_h2_profile.loc[
                (not cond_lcoe_leq) & cond_e_le_residue,
                f'Suministro {energy} Real'
            ] = df_h2_profile[f'Suministro {energy}']
            df_h2_profile.loc[
                (not cond_lcoe_leq) & ~cond_e_le_residue,
                f'Suministro {energy} Real'
            ] = residue_electro_aux

            df_h2_profile[f'Pérdida Energía {energy}'] = (
                    df_h2_profile[f'Suministro {energy}'] - df_h2_profile[f'Suministro {energy} Real'])

        else:
            df_h2_profile[f'Suministro {energy} Real'] = df_h2_profile[f'Suministro {energy}']
        df_h2_profile['Energía real utilizada'] += df_h2_profile[f'Suministro {energy} Real']
    
    df_h2_profile['Producción Hidrógeno'] = (
            df_h2_profile['Energía real utilizada'] * electrolyzer_config['electrolyzer_efficiency']
            * 1000 / constants['pci_h2'])

    return df_h2_profile


def calculate_operative_analysis(df_h2_profile, project_config, electrolyzer_config, energies):
    df_h2_profile['Emisiones CO2e (Grid)'] = 0

    df_h2_profile['Horas Operativas'] = (
            df_h2_profile['Energía real utilizada'] / electrolyzer_config['installed_capacity'])
    df_h2_profile['Horas Acumuladas'] = df_h2_profile['Horas Operativas'].cumsum()

    if 'grid' in energies.keys():
        df_h2_profile['Emisiones CO2e (Grid)'] = df_h2_profile['Suministro grid'] * energies['grid']['emission_factor']

    df_h2_profile['Consumo de agua'] = (
            df_h2_profile['Producción Hidrógeno'] * electrolyzer_config['water_consumption'] / 1000)
    df_h2_profile['Consumo eléctrico agua'] = (
            df_h2_profile['Consumo de agua'] * (
                project_config['electric_desalination_consumption'] +
                project_config['electric_pumping_consumption'] * project_config['coord_z'] / 100
            ) / 1000)

    if 'grid' in energies.keys():
        df_h2_profile['Emisiones agua'] = df_h2_profile['Consumo eléctrico agua'] * energies['grid']['emission_factor']
    else:
        df_h2_profile['Emisiones agua'] = df_h2_profile['Consumo eléctrico agua'] * constants['grid_emission_factor']

    return df_h2_profile


def calculate_financial_analysis(df_operative_analysis, project_config, electrolyzer_config, energies):

    operative_hours = df_operative_analysis['Horas Operativas'].sum()
    years_replacement_stack = electrolyzer_config['stack_life'] / operative_hours if operative_hours != 0 else 1000
    years_replacement_stack = np.floor(years_replacement_stack)
    yearly_h2_production = df_operative_analysis['Producción Hidrógeno'].sum()
    yearly_electro_degradation = 1 - electrolyzer_config['degradation']
    capex_electro = electrolyzer_config['installed_capacity'] * electrolyzer_config['capex'] * 1000
    opex_electro = capex_electro * electrolyzer_config['opex']

    df_fa = pd.DataFrame({'Año': [i for i in range(project_config['years_evaluation'] + 1)]})
    df_fa['Horas Anuales'] = operative_hours
    df_fa['Reemplazo'] = 0 

    cond_replacement_stack = df_fa['Año'].mod(int(years_replacement_stack)) == 0
    df_fa.loc[cond_replacement_stack, 'Reemplazo'] = 1
    
    df_fa['Horas reales'] = (
            df_fa.iloc[1:]['Horas Anuales'].cumsum() - (
                (df_fa[cond_replacement_stack]['Año']).repeat(years_replacement_stack) * operative_hours
            ).reset_index(drop=True).shift(1))  #

    df_fa['Producción de H2'] = (
            np.append(np.nan, np.tile(
                np.append(
                    [yearly_h2_production],
                    (np.array([yearly_electro_degradation] * int(years_replacement_stack - 1)).cumprod()
                     * yearly_h2_production)
                ), int(np.ceil(25 / years_replacement_stack))
            )[:25]))  #

    df_fa['CAPEX Electrolizador'] = 0
    df_fa.loc[0, 'CAPEX Electrolizador'] = capex_electro
    df_fa['OPEX Electrolizador'] = opex_electro
    df_fa.loc[0, 'OPEX Electrolizador'] = 0

    df_fa['CAPEX solar'] = 0
    df_fa['OPEX solar'] = 0
    df_fa['CAPEX eolic'] = 0
    df_fa['OPEX eolic'] = 0

    df_fa['Costo Energía solar'] = 0
    df_fa['Costo Energía eolic'] = 0
    df_fa['Costo Energía grid'] = 0
    
    for energy in energies.keys():
        if energy in ['solar', 'eolic'] and energies[energy]['cost_method'] == 'CAPEX y OPEX':
            capex_energy = energies[energy]['capex'] * energies[energy]['dimension'] * 1000
            opex_energy = capex_energy * energies[energy]['opex']
            df_fa.loc[0, f'CAPEX {energy}'] = capex_energy
            df_fa[f'OPEX {energy}'] = opex_energy
            df_fa.loc[0, f'OPEX {energy}'] = 0
        elif energy == 'grid' or energies[energy]['cost_method'] == 'LCOE':
            total_energy = df_operative_analysis[f'Suministro {energy}'].sum()
            df_fa[f'Costo Energía {energy}'] = total_energy * energies[energy]['lcoe']
            df_fa.loc[0, f'Costo Energía {energy}'] = 0
    
    df_fa['Reemplazo Stack'] = 0
    df_fa.loc[cond_replacement_stack, 'Reemplazo Stack'] = capex_electro * electrolyzer_config['cost_stack']
    df_fa.loc[0, 'Reemplazo Stack'] = 0

    total_water_consumption = df_operative_analysis['Consumo de agua'].sum()
    df_fa['Costo agua'] = (
            total_water_consumption * (
                project_config['water_cost'] + project_config['cost_pumping'] * project_config['dist_to_coast'] / 100))
    df_fa.loc[0, 'Costo agua'] = 0
    
    return df_fa


def calculate_financial_analysis_discounted(df_fa, project_config):
    df_fa_discounted = df_fa.copy()
    discount_rate = project_config['discount_rate']

    array_discount_denominator = np.append(
        1, np.array([(1 + discount_rate)] * project_config['years_evaluation']).cumprod())
    for column in ['Producción de H2', 'CAPEX Electrolizador', 'OPEX Electrolizador', 'CAPEX solar', 'OPEX solar',
                   'CAPEX eolic', 'OPEX eolic', 'Costo Energía solar', 'Costo Energía eolic', 'Costo Energía grid',
                   'Reemplazo Stack', 'Costo agua']:
        df_fa_discounted[column] /= array_discount_denominator

    total_sum = df_fa_discounted.sum()
    lcoh = total_sum / total_sum['Producción de H2']
    lcoh['Producción de H2'] = 0
    lcoh_pct = lcoh / lcoh.sum()

    return df_fa_discounted, total_sum, lcoh, lcoh_pct


def calculate_lcoh(lcoh):
    lcoh['Total'] = lcoh.iloc[5:].sum()

    results_lcoh_grouped = pd.Series()
    results_lcoh_grouped['CAPEX Electrolizador'] = lcoh['CAPEX Electrolizador'] + lcoh['Reemplazo Stack']
    results_lcoh_grouped['OPEX Electrolizador'] = lcoh['OPEX Electrolizador']
    results_lcoh_grouped['Energía'] = (
            lcoh['CAPEX solar'] + lcoh['OPEX solar'] + lcoh['CAPEX eolic'] + lcoh['OPEX eolic'] +
            lcoh['Costo Energía solar'] + lcoh['Costo Energía eolic'] + lcoh['Costo Energía grid'])
    results_lcoh_grouped['Agua'] = lcoh['Costo agua']
    results_lcoh_grouped['Total'] = results_lcoh_grouped.sum()

    return lcoh, results_lcoh_grouped


def calculate_results(df_oa, electrolyzer_config):
    df_gr = pd.DataFrame(index=['Variables'])
    df_gr['Consumo energético electrolizador'] = df_oa['Energía real utilizada'].sum()
    df_gr['Energía solar no utilizada'] = df_oa['Pérdida Energía solar'].sum()
    df_gr['Energía eólica no utilizada'] = df_oa['Pérdida Energía eolic'].sum()
    df_gr['Consumo energético agua'] = df_oa['Consumo eléctrico agua'].sum()
    df_gr['Producción de hidrógeno promedio'] = df_oa['Producción Hidrógeno'].sum()
    df_gr['Consumo de agua'] = df_oa['Consumo de agua'].sum()
    df_gr['Emisiones CO2e electrolizador'] = df_oa['Emisiones CO2e (Grid)'].sum()
    df_gr['Emisiones CO2e totales'] = df_gr['Emisiones CO2e electrolizador'] + df_oa['Emisiones agua'].sum()
    df_gr['Factor de emisión hidrógeno electrolizador'] = (
            df_gr['Emisiones CO2e electrolizador'] * 1000 / df_gr['Producción de hidrógeno promedio'])
    df_gr['Factor de emisión hidrógeno total'] = (
            df_gr['Emisiones CO2e totales'] * 1000 / df_gr['Producción de hidrógeno promedio'])
    df_oa['hour'] = df_oa.index.strftime('%H:%M')
    df_hourly_profile = df_oa.groupby('hour').agg({
        'Suministro solar Real': 'mean',
        'Suministro eolic Real': 'mean',
        'Suministro grid Real': 'mean'
    })
    df_hourly_profile['%'] = df_hourly_profile.sum(axis=1) / electrolyzer_config['installed_capacity']

    return df_gr.T, df_hourly_profile


def run_model(input_, energies_profile):
    # energies_profile == df_h2_profile == df_oa
    df_h2_profile = calculate_h2_profile(energies_profile, input_['electrolyzer'], input_['energies'])
    df_oa = calculate_operative_analysis(df_h2_profile, input_['project'], input_['electrolyzer'], input_['energies'])

    df_fa = calculate_financial_analysis(df_oa, input_['project'], input_['electrolyzer'], input_['energies'])

    df_fa_discounted, total_sum, df_lcoh, lcoh_pct = calculate_financial_analysis_discounted(df_fa, input_['project'])
    df_investment_operation = total_sum

    df_lcoh, df_lcoh_grouped = calculate_lcoh(df_lcoh)

    df_results, df_hourly_profile = calculate_results(df_oa, input_['electrolyzer'])

    res_calculation = {
        'Resultados Generales': df_results,
        'Costos de inversión y operación': df_investment_operation.iloc[5:],
        'LCOH': df_lcoh.iloc[5:],
        'LCOH Agrupado': df_lcoh_grouped,
        'Perfil horario de generación promedio (usada)': df_hourly_profile
    }

    return res_calculation, df_lcoh['Total']
