import numpy as np
import pandas as pd

PCI_H2 = 33.333 # kWh/kg H2
GRID_EMISSION_FACTOR = 0.3907

def calculate_h2_profile(electrolyzer_config, energies):

    energies_list = energies["energies_list"]

    dict_complementary_renewable = {"solar":"eolic",
                                    "eolic":"solar"}
    df_h2_profile = energies["energies_profile"]

    df_h2_profile["Suministro solar"] = 0
    df_h2_profile["Suministro eolic"] = 0
    df_h2_profile["Suministro renovable"] = 0
    df_h2_profile["Suministro grid"] = 0
    df_h2_profile["Total Electricidad"] = 0
    df_h2_profile["Aux Energia solar"] = 0
    df_h2_profile["Aux Energia eolic"] = 0
    df_h2_profile[f"Pérdida Energía solar"] = 0
    df_h2_profile[f"Pérdida Energía eolic"] = 0
    df_h2_profile[f"Suministro solar Real"] = 0
    df_h2_profile[f"Suministro eolic Real"] = 0
    df_h2_profile[f"Suministro grid Real"] = 0
    df_h2_profile["Energía real utilizada"] = 0
    
    

    for e in energies_list:
        if e in ["solar","eolic"]:
            df_h2_profile[f"Suministro {e}"] = (energies["energies_config"][e]["dimension"]*df_h2_profile[f"Factor Planta {e}"]).fillna(0)
            df_h2_profile["Suministro renovable"] += df_h2_profile[f"Suministro {e}"]#.round(2)
    #print(df_h2_profile)
    if "grid" in energies_list:
        grid_complement = electrolyzer_config["installed_capacity"]*energies["energies_config"]["grid"]["complement_percentage"]
        cond_grid_complement = df_h2_profile["Suministro renovable"] < grid_complement
        df_h2_profile.loc[cond_grid_complement, "Suministro grid"] = (grid_complement - df_h2_profile["Suministro renovable"])#.round(2)

    df_h2_profile["Total Electricidad"] = df_h2_profile["Suministro renovable"] + df_h2_profile["Suministro grid"]

    cond_total_ge_electro = df_h2_profile["Suministro renovable"] > electrolyzer_config["installed_capacity"]
    for e in energies_list:
        if e in ["solar","eolic"]:
            cond_e_le_electro = df_h2_profile[f"Suministro {e}"] < electrolyzer_config["installed_capacity"]
            df_h2_profile[f"Aux Energia {e}"] = df_h2_profile[f"Suministro {e}"]
            df_h2_profile.loc[(cond_total_ge_electro&cond_e_le_electro), f"Aux Energia {e}"] = df_h2_profile[f"Suministro {e}"]
            df_h2_profile.loc[(cond_total_ge_electro&~cond_e_le_electro), f"Aux Energia {e}"] = electrolyzer_config["installed_capacity"]
            
    for e in energies_list:
        if e in ["solar","eolic"]:
            if dict_complementary_renewable[e] in energies_list:
                cond_lcoe_leq = energies["energies_config"][e]["lcoe"] < energies["energies_config"][dict_complementary_renewable[e]]["lcoe"]
            else:
                cond_lcoe_leq = np.zeros(df_h2_profile.shape[0],dtype=bool)
            residue_electro_aux = electrolyzer_config["installed_capacity"] - df_h2_profile[f"Aux Energia {dict_complementary_renewable[e]}"]
            cond_e_le_residue = df_h2_profile[f"Suministro {e}"] < residue_electro_aux 
            df_h2_profile[f"Suministro {e} Real"] = df_h2_profile[f"Aux Energia {e}"]
            df_h2_profile.loc[(~cond_lcoe_leq&cond_e_le_residue), f"Suministro {e} Real"] = df_h2_profile[f"Suministro {e}"] 
            df_h2_profile.loc[(~cond_lcoe_leq&~cond_e_le_residue), f"Suministro {e} Real"] = residue_electro_aux

            df_h2_profile[f"Pérdida Energía {e}"] = df_h2_profile[f"Suministro {e}"] - df_h2_profile[f"Suministro {e} Real"]

        else:
            df_h2_profile[f"Suministro {e} Real"] = df_h2_profile[f"Suministro {e}"] 
        df_h2_profile["Energía real utilizada"] += df_h2_profile[f"Suministro {e} Real"]
    
    df_h2_profile["Producción Hidrógeno"] = df_h2_profile["Energía real utilizada"]*electrolyzer_config["electrolyzer_efficiency"]*1000/PCI_H2

    return df_h2_profile

def calculate_operative_analysis(df_h2_profile, project_config, electrolyzer_config, energies):

    energies_list = energies["energies_list"]
    
    df_h2_profile["Emisiones CO2e (Grid)"] = 0

    df_h2_profile["Horas Operativas"] = df_h2_profile["Energía real utilizada"]/electrolyzer_config["installed_capacity"]
    df_h2_profile["Horas Acumuladas"] = df_h2_profile["Horas Operativas"].cumsum()

    if "grid" in energies_list:
        df_h2_profile["Emisiones CO2e (Grid)"] = df_h2_profile["Suministro grid"]*energies["energies_config"]["grid"]["emission_factor"]

    df_h2_profile["Consumo de agua"] = df_h2_profile["Producción Hidrógeno"]*electrolyzer_config["water_consumption"]/1000
    #print("altura ", project_config["coord_z"])
    df_h2_profile["Consumo eléctrico agua"] = df_h2_profile["Consumo de agua"]*(project_config["electric_desalination_consumption"]+project_config["electric_pumping_consumption"]*project_config["coord_z"]/100)/1000

    if "grid" in energies["energies_config"].keys():
        df_h2_profile["Emisiones agua"] = df_h2_profile["Consumo eléctrico agua"]*energies["energies_config"]["grid"]["emission_factor"]
    else:
        df_h2_profile["Emisiones agua"] = df_h2_profile["Consumo eléctrico agua"]*GRID_EMISSION_FACTOR

    return df_h2_profile

def calculate_financial_analysis(df_operative_analysis, project_config, electrolyzer_config, energies):

    energies_list = energies["energies_list"]

    operative_hours = df_operative_analysis["Horas Operativas"].sum()
    years_replacement_stack = electrolyzer_config["stack_life"]/operative_hours if operative_hours != 0 else 1000
    years_replacement_stack = np.ceil(years_replacement_stack)
    yearly_h2_production = df_operative_analysis["Producción Hidrógeno"].sum()
    yearly_electro_degradation = 1- electrolyzer_config["degradation"]
    capex_electro = electrolyzer_config["installed_capacity"]*electrolyzer_config["capex"]*1000
    opex_electro = capex_electro*electrolyzer_config["opex"]

    df_fa = pd.DataFrame({"Año":[i for i in range(project_config["years_evaluation"] + 1)]})

    df_fa["Horas Anuales"] = operative_hours
    
    df_fa["Reemplazo"] = 0 

    #print("REPLACEMENT", years_replacement_stack)
    cond_replacement_stack = df_fa["Año"].mod(int(years_replacement_stack)) == 0
    df_fa.loc[cond_replacement_stack,"Reemplazo"] = 1
    
    df_fa["Horas reales"] = df_fa.iloc[1:]["Horas Anuales"].cumsum()-((df_fa[cond_replacement_stack]["Año"]).repeat(years_replacement_stack)*operative_hours).reset_index(drop=True).shift(1)

    df_fa["Producción de H2"] = np.append(np.nan,np.tile(np.append([yearly_h2_production],(np.array([yearly_electro_degradation]*int(years_replacement_stack-1)).cumprod()*yearly_h2_production)),int(np.ceil(25/years_replacement_stack)))[:25])
    	
    df_fa["CAPEX Electrolizador"] = 0
    df_fa.loc[0,"CAPEX Electrolizador"] = capex_electro
    df_fa["OPEX Electrolizador"] = opex_electro
    df_fa.loc[0,"OPEX Electrolizador"] = 0


    df_fa["CAPEX solar"] = 0
    df_fa["OPEX solar"] = 0
    df_fa["CAPEX eolic"] = 0
    df_fa["OPEX eolic"] = 0

    df_fa["Costo Energía solar"] = 0
    df_fa["Costo Energía eolic"] = 0
    df_fa["Costo Energía grid"] = 0
    
    for e in energies_list:
        if e in ["solar","eolic"] and energies["energies_config"][e]["cost_method"] == "CAPEX y OPEX":
            capex_energy = energies["energies_config"][e]["capex"]*energies["energies_config"][e]["dimension"]*1000
            opex_energy = capex_energy*energies["energies_config"][e]["opex"]
            df_fa.loc[0,f"CAPEX {e}"] = capex_energy
            df_fa[f"OPEX {e}"] = opex_energy
            df_fa.loc[0,f"OPEX {e}"] = 0
        elif e=="grid" or energies["energies_config"][e]["cost_method"] == "LCOE":
            total_energy = df_operative_analysis[f"Suministro {e}"].sum()
            df_fa[f"Costo Energía {e}"] = total_energy*energies["energies_config"][e]["lcoe"]
            df_fa.loc[0,f"Costo Energía {e}"] = 0
    
    df_fa["Reemplazo Stack"] = 0
    df_fa.loc[cond_replacement_stack, "Reemplazo Stack"] = capex_electro*electrolyzer_config["cost_stack"]
    df_fa.loc[0, "Reemplazo Stack"] = 0

    total_water_consumption = df_operative_analysis["Consumo de agua"].sum()
    df_fa["Costo agua"] = total_water_consumption*(project_config["water_cost"] + 
                                                   project_config["cost_pumping"]*project_config["dist_to_coast"]/100)
    df_fa.loc[0,"Costo agua"] = 0
    #if "grid" in energies["energies_config"].keys():
    #    df_fa["Costo energía para agua"] = df_operative_analysis["Consumo eléctrico agua"].sum()*energies["energies_config"]["grid"]["lcoe"]
    #else:
    #    df_fa["Costo energía para agua"] = df_operative_analysis["Consumo eléctrico agua"].sum()*project_config["grid_energy_cost"]
    #df_fa.loc[0,"Costo energía para agua"] = 0
    
    return df_fa

def calculate_financial_analysis_discounted(df_fa, project_config):

    df_fa_discounted = df_fa.copy()
    discount_rate = project_config["discount_rate"]

    array_discount_denominator = np.append(1,np.array([(1+discount_rate)]*project_config["years_evaluation"]).cumprod())
    df_fa_discounted["Producción de H2"] = df_fa_discounted["Producción de H2"]/array_discount_denominator
    df_fa_discounted["CAPEX Electrolizador"] = df_fa_discounted["CAPEX Electrolizador"]/array_discount_denominator
    df_fa_discounted["OPEX Electrolizador"] = df_fa_discounted["OPEX Electrolizador"]/array_discount_denominator
    df_fa_discounted["CAPEX solar"] = df_fa_discounted["CAPEX solar"]/array_discount_denominator
    df_fa_discounted["OPEX solar"] = df_fa_discounted["OPEX solar"]/array_discount_denominator
    df_fa_discounted["CAPEX eolic"] = df_fa_discounted["CAPEX eolic"]/array_discount_denominator
    df_fa_discounted["OPEX eolic"] = df_fa_discounted["OPEX eolic"]/array_discount_denominator
    df_fa_discounted["Costo Energía solar"] = df_fa_discounted["Costo Energía solar"]/array_discount_denominator
    df_fa_discounted["Costo Energía eolic"] = df_fa_discounted["Costo Energía eolic"]/array_discount_denominator
    df_fa_discounted["Costo Energía grid"] = df_fa_discounted["Costo Energía grid"]/array_discount_denominator
    df_fa_discounted["Reemplazo Stack"] = df_fa_discounted["Reemplazo Stack"]/array_discount_denominator
    df_fa_discounted["Costo agua"] = df_fa_discounted["Costo agua"]/array_discount_denominator
    #df_fa_discounted["Costo energía para agua"] = df_fa_discounted["Costo energía para agua"]/array_discount_denominator

    total_sum = df_fa_discounted.sum()
    lcoh = total_sum/total_sum["Producción de H2"]
    lcoh["Producción de H2"] = 0
    lcoh_pct = lcoh/lcoh.sum()

    #print(df_fa_discounted.iloc[0:20, 12:20])

    return df_fa_discounted, total_sum, lcoh, lcoh_pct

def calculate_results(df_oa, electrolyzer_config):

    df_gr = pd.DataFrame(index=["Variables"])
    df_gr["Consumo energético electrolizador"] = df_oa["Energía real utilizada"].sum()
    df_gr["Energía solar no utilizada"] = df_oa["Pérdida Energía solar"].sum()
    df_gr["Energía eólica no utilizada"] = df_oa["Pérdida Energía eolic"].sum()
    df_gr["Consumo energético agua"] = df_oa["Consumo eléctrico agua"].sum()
    df_gr["Producción de hidrógeno promedio"] = df_oa["Producción Hidrógeno"].sum()
    df_gr["Consumo de agua"] = df_oa["Consumo de agua"].sum()
    df_gr["Emisiones CO2e electrolizador"] = df_oa["Emisiones CO2e (Grid)"].sum()
    df_gr["Emisiones CO2e totales"] = df_gr["Emisiones CO2e electrolizador"] + df_oa["Emisiones agua"].sum()
    df_gr["Factor de emisión hidrógeno electrolizador"] = df_gr["Emisiones CO2e electrolizador"]*1000/df_gr["Producción de hidrógeno promedio"]
    df_gr["Factor de emisión hidrógeno total"] = df_gr["Emisiones CO2e totales"]*1000/df_gr["Producción de hidrógeno promedio"]
    #print(df_oa.index[:3])
    df_oa["hour"] = df_oa.index.str.slice(5,11)
    #print(df_oa["hour"][:3])
    df_hourly_profile = df_oa.groupby("hour").agg({"Suministro solar Real":"mean",
                                                   "Suministro eolic Real":"mean",
                                                   "Suministro grid Real":"mean" })
    df_hourly_profile["%"] = df_hourly_profile.sum(axis=1)/electrolyzer_config["installed_capacity"]

    return df_gr.T, df_hourly_profile 

def calculate_lcoh(lcoh):

    results_lcoh = lcoh.copy()
    results_lcoh["Total"] = lcoh.iloc[5:].sum()
    #print(lcoh)

    results_lcoh_grouped = pd.Series()
    results_lcoh_grouped["CAPEX Electrolizador"] = lcoh["CAPEX Electrolizador"] + lcoh["Reemplazo Stack"]
    results_lcoh_grouped["OPEX Electrolizador"] = lcoh["OPEX Electrolizador"]
    results_lcoh_grouped["Energía"] = lcoh["CAPEX solar"] + lcoh["OPEX solar"] + lcoh["CAPEX eolic"] + lcoh["OPEX eolic"] +\
                                      lcoh["Costo Energía solar"] + lcoh["Costo Energía eolic"] + lcoh["Costo Energía grid"]
    results_lcoh_grouped["Agua"] = lcoh["Costo agua"] #+ lcoh["Costo energía para agua"]
    results_lcoh_grouped["Total"] = results_lcoh_grouped.sum()
    
    return results_lcoh, results_lcoh_grouped

def run_model(input_calculator):

    """
    input: {
        "energies"{
            "energies_config"{},
            "energies_list": List()
            "energies_profile": DataFrame()
        }
    }

    """
    
    df_h2_profile = calculate_h2_profile(input_calculator["electrolyzer_config"], input_calculator["energies"])
    
    df_oa = calculate_operative_analysis(df_h2_profile, input_calculator["project_config"], input_calculator["electrolyzer_config"], input_calculator["energies"])
    #print(df_oa["Suministro eolic Real"].sum())
    #print(df_oa.iloc[:15,12:16])
    df_fa = calculate_financial_analysis(df_oa, input_calculator["project_config"], input_calculator["electrolyzer_config"], input_calculator["energies"])
    #print(df_fa.iloc[:15,0:4])
    df_fa_discounted, total_sum, lcoh, lcoh_pct = calculate_financial_analysis_discounted(df_fa, input_calculator["project_config"])
    #print(df_fa_discounted.iloc[:20,12:16])
    df_investment_operation = total_sum

    df_lcoh, df_lcoh_grouped = calculate_lcoh(lcoh)

    df_results, df_hourly_profile = calculate_results(df_oa, input_calculator["electrolyzer_config"])

    #print(df_hourly_profile.iloc[10:])
    res_calculation = {
        "Resultados Generales": df_results,
        "Costos de inversión y operación": df_investment_operation.iloc[5:],
        "LCOH": df_lcoh.iloc[5:],
        "LCOH Agrupado": df_lcoh_grouped,
        "Perfil horario de generación promedio (usada)": df_hourly_profile
    }

    return res_calculation, df_lcoh.loc["Total"]
