from datetime import date, datetime, timedelta, timezone

import json
import pandas as pd

PROPORTION_ENERGIES_ELECTROLYZER = 1.5

def create_generic_dts(initial_day=datetime(2015,1,1), n=8760):

    list_dts_str = []

    for i in range(n):
        list_dts_str.append(str(initial_day + timedelta(hours=i))[5:16])

    return list_dts_str

def format_request_energy_explorer(energy_config, project_config):

    if "profile_template" in energy_config.keys():
        filename_template = "config/templates/"+energy_config["profile_template"]
        dict_template_req = json.load(open(filename_template))
    else:
        dict_template_req = energy_config

    if "lon" in project_config["position"].keys():
        dict_template_req["position"][0]["lon"] = project_config["position"]["lon"]
    if "lat" in project_config["position"].keys():
        dict_template_req["position"][0]["lat"] = project_config["position"]["lat"]

    return json.dumps(dict_template_req)


def compare_dicts(dict_ref, dict_input, str_trace=""):

    for k,v in dict_ref.items():
        if k in dict_input.keys():
            if isinstance(v, dict):
                compare_dicts(dict_ref[k], dict_input[k], str_trace=str_trace+f"{k}_")
            elif isinstance(v, list):
                limit_i = v[0]
                limit_s = v[1]
                if (limit_i!= None and limit_i > dict_input[k]) or (limit_s!= None and limit_s < dict_input[k]):
                    raise ValueError(f"Value input {str_trace}{k} {dict_input[k]} out of bounds for range: [{limit_i}, {limit_s}]")


def validate_inputs(inputs):

    filename_limits = "config/input_limits.json"
    dict_limits = json.load(open(filename_limits))[str(inputs["project"]["year"])]

    compare_dicts(dict_limits, inputs)

def format_inputs(filename, input_type="input", points=None):

    """
    output: {
        "energies"{
            "energies_config"{},
            "energies_list": List()
            "energies_profile": DataFrame()
        }
    }
    """

    inputs = json.load(open(filename))
    dict_inputs_formatted = {}

    if input_type=="input":
        
        inputs["project"]["position"] = {}
        inputs["project"]["position"]["lat"] = points[0]
        inputs["project"]["position"]["lon"] = points[1]

        validate_inputs(inputs)

        list_energies = []

        # format energies input
        for energy in inputs["energies"]["energies_config"]:
            list_energies.append(energy)
            if energy in ["eolic", "solar"]:
                inputs["energies"]["energies_config"][energy]["profile_config"] = format_request_energy_explorer(inputs["energies"]["energies_config"][energy], inputs["project"])
        
        inputs["energies"]["energies_list"] = list_energies
        
        dict_inputs_formatted["project_config"] = inputs["project"]
        dict_inputs_formatted["electrolyzer_config"] = inputs["electrolyzer"]
        dict_inputs_formatted["energies"] = inputs["energies"]
    elif input_type == "repeater":
        dict_inputs_formatted["points"] = inputs["execution"]["points"]
        dict_inputs_formatted["output_with_energy_profiles"] = inputs["execution"]["output_with_energy_profiles"]
        dict_inputs_formatted["output_with_energies_balance"] = inputs["execution"]["output_with_energies_balance"]
        dict_inputs_formatted["b_iterations"] = 1
        if inputs["execution"]["output_with_energies_balance"]:
            dict_inputs_formatted["b_iterations"] = 12
        dict_inputs_formatted["output_format"] = inputs["execution"]["output_format"]

    return dict_inputs_formatted


def process_inputs_hv_calculator(inputs_formatted, iteration=0):

    """
    input: {
        "energies"{
            "energies_config"{},
            "energies_list": List()
            "energies_profile": DataFrame()
        }
    }
    """

    list_dts = []
    factor_eolic = None
    factor_solar = None

    for energy in inputs_formatted["energies"]["energies_list"]:  
        if energy in ["eolic", "solar"]:
            inputs_formatted["energies"]["energies_profile"][f"Factor Planta {energy}"] = inputs_formatted["energies"]["energies_profile"][f"Perfil de Energía - {energy}"]\
                                                                                        /inputs_formatted["energies"]["energies_config"][energy]["total_power_profile"]

            if inputs_formatted["energies"]["energies_config"][energy]["cost_method"] == "CAPEX y OPEX":
                inputs_formatted["energies"]["energies_config"][energy]["lcoe"] =   (inputs_formatted["energies"]["energies_config"][energy]["capex"]*\
                                                                                     inputs_formatted["energies"]["energies_config"][energy]["dimension"]*1000+\
                                                                                        (inputs_formatted["energies"]["energies_config"][energy]["dimension"]*\
                                                                                        inputs_formatted["energies"]["energies_config"][energy]["capex"]*1000*\
                                                                                        inputs_formatted["energies"]["energies_config"][energy]["opex"]/0.08)*\
                                                                                        (1-1/(1+inputs_formatted["project_config"]["discount_rate"])**inputs_formatted["project_config"]["years_evaluation"]))/\
                                                                                    (inputs_formatted["energies"]["energies_config"][energy]["dimension"]*\
                                                                                     inputs_formatted["energies"]["energies_profile"][f"Factor Planta {energy}"].mean()*8760/\
                                                                                     inputs_formatted["project_config"]["discount_rate"]*\
                                                                                        (1-1/(1+inputs_formatted["project_config"]["discount_rate"])**\
                                                                                        inputs_formatted["project_config"]["years_evaluation"]))

            #print(inputs_formatted["energies"]["energies_profile"][f"Perfil de Energía - {energy}"].head(10))

    if inputs_formatted["energies"]["energies_profile"].shape[0] != 0:
        list_dts = inputs_formatted["energies"]["energies_profile"].index.astype(str).str.slice(5,16)
        #print(inputs_formatted["energies"]["energies_profile"].index[:3])
        #print(list_dts[:3])
        inputs_formatted["energies"]["energies_profile"].index = list_dts
    
    if len(list_dts) == 0:
       list_dts = create_generic_dts()
       inputs_formatted["energies"]["energies_profile"] = pd.DataFrame(index=list_dts)

    if iteration > 0 and "eolic" in inputs_formatted["energies"]["energies_list"] and "solar" in inputs_formatted["energies"]["energies_list"]:
        
        factor_eolic = round((1 - iteration/10 + 0.1)*100)
        factor_solar = round((iteration/10 - 0.1)*100)
        
        dimension_eolic = 0.1*factor_eolic
        dimension_solar = 0.1*factor_solar
        dimension_electrolyzer = (dimension_eolic + dimension_solar)/PROPORTION_ENERGIES_ELECTROLYZER

        inputs_formatted["energies"]["energies_config"]["eolic"]["dimension"] = dimension_eolic
        inputs_formatted["energies"]["energies_config"]["solar"]["dimension"] = dimension_solar

        inputs_formatted["electrolyzer_config"]["installed_capacity"] = dimension_electrolyzer

        #print("Dimensions energies", dimension_eolic, dimension_solar, dimension_electrolyzer) 

    return inputs_formatted, (factor_eolic, factor_solar)
    

def save_output(inputs_formatted, execution_inputs_formatted, res_calculation , list_lcoh, filename="/tmp/lcoh_model"):

    final_filename = f"{filename}.{execution_inputs_formatted['output_format']}"
    df_lcoh = pd.DataFrame()
    #print(execution_inputs_formatted)
    if execution_inputs_formatted["output_with_energies_balance"]:
        df_lcoh = pd.DataFrame(list_lcoh, columns=["eolic %", "solar %","LCOH"])
    if execution_inputs_formatted["output_format"] == "xlsx":
        with pd.ExcelWriter(final_filename, engine='xlsxwriter') as writer1:
            for config in inputs_formatted:
                #print(config)
                if config == "project_config":
                    inputs_formatted[config]["position"] = str(inputs_formatted[config]["position"])
                if config != "energies":
                    df_aux = pd.DataFrame(inputs_formatted[config], index=[0])
                    df_aux.to_excel(writer1, sheet_name=f"{config}")
                elif execution_inputs_formatted["output_with_energy_profiles"]:
                    df_aux = inputs_formatted[config]["energies_profile"]
                    df_aux.to_excel(writer1, sheet_name=f"{config}_profiles")
            for res in res_calculation:
                res_calculation[res].to_excel(writer1, sheet_name=res[:30])
            if df_lcoh.shape[0] != 0:
                df_lcoh.to_excel(writer1, sheet_name="LCOH energy balance", index=False) 
    elif execution_inputs_formatted["output_format"] == "json":
        for k  in res_calculation:
            res_calculation[k] = res_calculation[k].to_dict()
        if df_lcoh.shape != 0:
            res_calculation["LCOH energy balance"] = df_lcoh.to_dict()
        with open(final_filename, 'w') as f:
            json.dump(res_calculation, f, ensure_ascii=False)